<?php

/**
 * Class CommentsController_bwg
 */
class CommentsController_bwg {
  /**
   * @var $model
   */
  private $model;
  /**
   * @var $view
   */
  private $view;
  /**
   * @var string $page
   */
  private $page;
  /**
   * @var string $bulk_action_name
   */
  private $bulk_action_name;
  /**
   * @var int $items_per_page
   */
  private $items_per_page = 20;
  /**
   * @var array $actions
   */
  private $actions = array();

  public function __construct() {
    $this->model = new CommentsModel_bwg();
    $this->view = new CommentsView_bwg();

    $this->page = WDWLibrary::get('page');

    $this->actions = array(
      'publish' => array(
        'title' => __('Publish', BWG()->prefix),
        $this->bulk_action_name => __('published', BWG()->prefix),
      ),
      'unpublish' => array(
        'title' => __('Unpublish', BWG()->prefix),
        $this->bulk_action_name => __('unpublished', BWG()->prefix),
      ),
      'delete' => array(
        'title' => __('Delete', BWG()->prefix),
        $this->bulk_action_name => __('deleted', BWG()->prefix),
      ),
    );

    $user = get_current_user_id();
    $screen = get_current_screen();
    $option = $screen->get_option('per_page', 'option');
    $this->items_per_page = get_user_meta($user, $option, true);

    if ( empty ( $this->items_per_page) || $this->items_per_page < 1 ) {
      $this->items_per_page = $screen->get_option( 'per_page', 'default' );
    }
  }

  /**
   * Execute.
   */
  public function execute() {
    $task = WDWLibrary::get('task');
    $id = (int) WDWLibrary::get('current_id', 0, 'inval');
    if ( $task != 'display' && method_exists($this, $task) ) {
      check_admin_referer(BWG()->nonce, BWG()->nonce);
      $action = WDWLibrary::get('bulk_action', -1);
      if ( $action != -1 ) {
        $this->bulk_action($action);
      }
      else {
        $this->$task($id);
      }
    }
    else {
      $this->display();
    }
  }

  /**
   * Display.
   */
  public function display() {
    // Set params for view.
    $params = array();
    $params['page'] = $this->page;
    $params['page_title'] = __('Comments', BWG()->prefix);
    $params['actions'] = $this->actions;
    $params['filters_config'] = $this->filters_config();
    $params['order'] = WDWLibrary::get('order', 'desc');
    $params['orderby'] = WDWLibrary::get('orderby', 'date');
    // To prevent SQL injections.
    $params['order'] = ($params['order'] == 'desc') ? 'desc' : 'asc';
    if ( !in_array($params['orderby'], array( 'name', 'mail', 'comment' )) ) {
      $params['orderby'] = 'date';
    }
    $params['items_per_page'] = $this->items_per_page;
    $page = WDWLibrary::get('paged', 1, 'intval');
    $page_num = $page ? ($page - 1) * $params['items_per_page'] : 0;
    $params['page_num'] = $page_num;
    $params['search'] = WDWLibrary::get('s');
    $params['total'] = $this->model->total($params);
    $params['rows_data'] = $this->model->get_rows_data($params);
    $this->view->display($params);
  }

  public function filters() {
    $filter = WDWLibrary::get('filter');
    $filter_action = WDWLibrary::get('filter_action') . '&' . http_build_query($filter, '', '&');
    $redirect = add_query_arg(array(
                                'page' => $this->page,
                                'task' => 'display',
                                'filter_action' => $filter_action,
                              ), admin_url('admin.php'));
    WDWLibrary::redirect($redirect);
  }

  private function filters_config( $params = array() ) {
    $gallery_id = 0;
    $filters = array(
      'filters' => array(
        'gallery' => array(
          'label' => __('Filter by gallery', BWG()->prefix),
          'items' => $this->model->get_galleries()
        ),
        'image' => array(
          'label' => __('Filter by image', BWG()->prefix),
          'items' => $this->model->get_images($gallery_id)
        ),
      )
    );
    return $filters;
  }

  /**
   * Bulk actions.
   *
   * @param $task
   */
  public function bulk_action($task) {
    $message = 0;
    $successfully_updated = 0;
    $url_arg = array('page' => $this->page,'task' => 'display');

    $check = WDWLibrary::get('check');
    $all = WDWLibrary::get('check_all_items');
    $all = ($all == 'on' ? TRUE : FALSE);

    if ( method_exists($this, $task) ) {
      if ( $all ) {
        $message = $this->$task(0, TRUE, TRUE);
        $url_arg['message'] = $message;
      }
      else {
        if ( $check ) {
          foreach ( $check as $form_id => $item ) {
            $message = $this->$task($form_id, TRUE);
            if ( $message != 2 ) {
              // Increase successfully updated items count, if action doesn't failed.
              $successfully_updated++;
            }
          }
        }
        if ( $successfully_updated ) {
          $bulk_action = $this->bulk_action_name;
          $message = sprintf(_n('%s item successfully %s.', '%s items successfully %s.', $successfully_updated, BWG()->prefix), $successfully_updated, $this->actions[$task][$bulk_action]);
        }
        $key = ($message === 2 ? 'message' : 'msg');
        $url_arg[$key] = $message;
      }
    }

    WDWLibrary::redirect(add_query_arg($url_arg, admin_url('admin.php')));
  }

  /**
   * Delete form by id.
   *
   * @param      $id
   * @param bool $bulk
   * @param bool $all
   *
   * @return int
   */
  public function delete( $id, $bulk = FALSE, $all = FALSE ) {
    global $wpdb;
    $search = WDWLibrary::get('s');
    $prepareArgs = array();
    $where = '';
    if( $all ) {
      if($search != '') {
        $where = ' WHERE `name` LIKE %s';
        $prepareArgs[] = "%" . $search . "%";
      }
    } else {
        $where = ' WHERE id=%d';
        $prepareArgs[] = $id;
    }

    if( !empty($prepareArgs) ) {
      $delete = $wpdb->query($wpdb->prepare('DELETE FROM `' . $wpdb->prefix . 'bwg_image_comment`' . $where, $prepareArgs));
    } else {
      $delete = $wpdb->query('DELETE FROM `' . $wpdb->prefix . 'bwg_image_comment`' . $where);
    }

    if ( $delete ) {
      if ( $all ) {
        $message = 5;
      }
      else {
        $message = 3;
      }
    }
    else {
      $message = 2;
    }
    if ( $bulk ) {
      return $message;
    }
    WDWLibrary::redirect( add_query_arg( array('page' => $this->page, 'task' => 'display', 'message' => $message), admin_url('admin.php') ) );
  }

  /**
   * Publish by id.
   *
   * @param $id
   * @param bool $bulk
   * @param bool $all
   *
   * @return int
   */
  public function publish( $id, $bulk = FALSE, $all = FALSE ) {
    global $wpdb;
    $search = WDWLibrary::get('s');
    $where = '';
    $prepareArgs = array();
    if( $all ) {
      if($search != '') {
        $where = ' WHERE `name` LIKE %s';
        $prepareArgs[] = "%" . $search . "%";
      }
    } else {
      $where = ' WHERE id=%d';
      $prepareArgs[] = $id;
    }
    if( !empty($prepareArgs) ) {
      $updated = $wpdb->query($wpdb->prepare('UPDATE `' . $wpdb->prefix . 'bwg_image_comment` SET published=1' . $where, $prepareArgs));
    } else {
      $updated = $wpdb->query('UPDATE `' . $wpdb->prefix . 'bwg_image_comment` SET published=1' . $where);
    }
    if ( $updated !== FALSE ) {
      $message = 9;
    }
    else {
      $message = 2;
    }

    if ( $bulk ) {
      return $message;
    }
    else {
      WDWLibrary::redirect(add_query_arg(array(
                                                  'page' => $this->page,
                                                  'task' => 'display',
                                                  'message' => $message,
                                                ), admin_url('admin.php')));
    }
  }

  /**
   * Unpublish by id.
   *
   * @param $id
   * @param bool $bulk
   * @param bool $all
   *
   * @return int
   */
  public function unpublish( $id, $bulk = FALSE, $all = FALSE ) {
    global $wpdb;
    $search = WDWLibrary::get('s');
    $where = '';
    $prepareArgs = array();
    if( $all ) {
      if($search != '') {
        $where = ' WHERE `name` LIKE %s';
        $prepareArgs[] = "%" . $search . "%";
      }
    } else {
      $where = ' WHERE id=%d';
      $prepareArgs[] = $id;
    }
    if( !empty($prepareArgs) ) {
      $updated = $wpdb->query($wpdb->prepare('UPDATE `' . $wpdb->prefix . 'bwg_image_comment` SET published=0' . $where, $prepareArgs));
    } else {
      $updated = $wpdb->query('UPDATE `' . $wpdb->prefix . 'bwg_image_comment` SET published=0' . $where);
    }

    if ( $updated !== FALSE ) {
      $message = 10;
    }
    else {
      $message = 2;
    }

    if ( $bulk ) {
      return $message;
    }
    else {
      WDWLibrary::redirect(add_query_arg(array(
                                                  'page' => $this->page,
                                                  'task' => 'display',
                                                  'message' => $message,
                                                ), admin_url('admin.php')));
    }
  }
}
